/*
XCraft Data Conversion Utility
Gregory Alan Hildstrom
Hildstrom Engineering
http://www.hildstrom.com
*/
#include <iostream>
#include <fstream>
#include <map>
#include <vector>
using namespace std;

/******Byte Ordering***************************
Hello. Thank you for the documentation and the example hex dumps. These
cleared up my understanding a little bit. You may want to explicitly state
that all of the 2-byte pairs must be reordered to read this data on i686
architectures. This gets back to some of my questions that have remained
unanswered regarding high-byte-first vs low-byte-first and bit ordering.

Notice the packet size bytes:
Linux: 8805
Yours: 0598
Notice the destip bytes:
Linux: a8c0 f201
Yours: C0A8 01F2
Notice the time tag bytes:
Linux: fcff
Yours: FFFC

Linux hexdump of SEA_008_BK2.raw
0000000 8805 0000 a8c0 f201 a8c0 0201 d4a8 0700
0000010 c000 5c01 0000 b900 fcff 0000 fdff 0000
0000020 feff 5806 ffff d3fa 7800 04fd 7900 5801

Your hex for brick 11
0000000 0598 0000 C0A8 01F2 C0A8 010B A8D4 0007
0000010 00C0 0160 0000 000E FFFC 0000 FFFD 0000
0000020 FFFE 0014 FFFF D485 00BC 0160 00BD FF68

Each 2-byte pair must be reordered on i686 architectures. Byte 1 should be
swapped with byte 2 in order for the numbers to be interpreted properly.
The example time tag is stored natively in the binary file as fcff. The
first byte fc must be swapped with the second byte ff. The correct
resultant time tag is fffc. Please include this information in your
documentation. Thank you for all of your help.

Gregory Alan Hildstrom
Hildstrom Engineering
http://www.hildstrom.com
NSWCCD Code 653
*************************************************/

/******DAS Brick File Decode*********************
(Except Brick 1 and Bricks 29 and greater)

	Almost all of the brick files recorded through the DAS server
	have the same format. This format is a concatenation of Brick
	UDP packets into a file. Once one packet is decoded out of the
	file the rest should fall into place. The following is a brief
	description of the UDP packet and an example of how to decode it.

Packet Format

	The packet is broken into two sections; the header and the data
	body. The header contains information that can be used to decode
	the data body. An example of a packet header in hexadecimal is:
	0598 0000 C0A8 01F2 C0A8 010B A8D4 0007 00C0 0160 0000 000E.

Packet Header:
	Name Size Size Bytes Example
	Packet Size WORD 2 Bytes 0598
	Flags WORD 2 Bytes 0000
	DestIP DWORD 4 bytes C0A8 01F2
	SrcIP DWORD 4 bytes C0A8 010B
	Rx Port WORD 2 Bytes A8D4
	FmtID (Data type) WORD 2 Bytes 0007
	NodeID WORD 2 Bytes 00C0
	Pair Count WORD 2 Bytes 0160
	Packet Count DWORD 4 bytes 0000 000E

Packet Size:
	This number gives the size of the entire packet including the header
	in bytes. The packet will not exceed 1500 bytes and is typically
	around 1450 bytes. With this information it is possible for the user
	to locate the next packet in the file. From the example header the
	Packet size in Hex 0598 is a packet length of 1432 bytes.

Flags:
	This number may possible give error flags or the like. At this time it is not used.

DestIP:
	This number is the IP address where the UDP packet is heading. Most
	likely the DAS Server. It is in a binary coded decimal. From the example
	header: C0A8 01F2 or C0.A8.01.F2 which is 192.168.1.242.

SrcIP:
	This is the IP address of the Brick that is sending the packet. It is in
	a binary coded decimal. From the example header: C0A8 010B or C0.A8.01.0B which is 192.168.1.11.

Rx Port:
	This is the UDP port the brick is sending too. From the example header: A8D4 is port number 43220.

FmtID:
	This is the type of data being sent in the packet. 7 is analog, 8 is NMEA,
	9 is MK27, 10 is RCS, and 11 is MCS 7 (Numbers might change for RSC and MCS 7).
	This is only used to decode information from the Serial and RCS bricks. From
	the example header 0007 is data type 7

NodeID:
	This number give some information that is not known to us at this time.

Pair Count:
	This gives the number of data pairs located with in the packet. A data pair
	is defined as a data tag and the data. The data tag and the data will be defined
	in the data body section later in this document. The pair count also includes
	the time tag and data. From the example header 0160 is 352 data pairs in the packet.

Packet Count:
	This gives the UDP packet number associated with the data packet. This number should
	be sequential with the packets around it. If the packet count is not sequential it
	means there was a packet lost. From the example header 0000 000E is the 14th packet
	sent from the brick.

Data Body:

	The data body can be broken down into two components. The first component is the
	Data Tag and the second is the data.

Data Tags
	The data tag is two bytes that label the data. These two bytes could be represented
	as a signed or unsigned integer because they are basically used for looking up the
	channel information. There are four data tags that are special because they indicate
	the brick time associated with the data they are the time tags in hexadecimal:
	FFFC, FFFD, FFFE, and FFFF.

Data
	The data is in a two byte signed integer. This is true for all data except the time
	data after the time tags. The time data is in four two byte unsigned integers that
	must be combined to produce a 64 bit time. The time is in micro seconds from a reset
	time command, or a brick boot-up.

Length of Data Body
	The length of the data body can be determined by the Pair Count value. The pair count
	includes the data tag and the data. Therefore, if one was to count the data tag  WORDs,
	and the data WORDs, when the count reached 1 plus twice the pair count a new packet
	should have been reached.

Matthew Powell
Naval Surface Warfare Center Carderock
Code 5500 Seakeeping
*************************************************/

/******XCRAFT PACKET DESCRIPTION*******************
Greg,

   I will try to get you what you need to decode the files to the best
of my ability. As of right now nothing is formally written down. I am
however working on a document that should give a detailed description.
To get you started a brief description.

Brick 2 (BK2) through Brick 28 (BK28) all have the same format; Brick 1
and Brick 29 will have unique formats, which I can not explain quickly
and will provide to you as soon as I get them written up.

BK2 through BK28 files are made up of appended packets. Once you get the
first packet everything else is just a repeat. The packets have the
following format.
PacketSize: WORD; 2 Bytes
Flags: WORD; 2 Bytes
DestIP: DWORD; 4 Bytes
SrcIP: DWORD; 4 Bytes
Port: WORD; 2 Bytes
FmtID: WORD; 2 Bytes
NodeID: WORD; 2 Bytes
PairCount: WORD; 2 Bytes (Pair Count is to include DataTag and Data, and
it is the total number of pairs in the packet's DataArray including
time)
PacketCount: DWORD; 4 Bytes (Should be sequential within a file, if not
a packet(s) was(were) lost)
DataArray:
The data in the data array is broken into two pieces the data tag and
the data. The data tags describe the data that follows (example FFFC is
the time's(8 bytes) high 2 bytes FFFD time's 2nd high 2 bytes, FFFE,
time's 3rd high 2 bytes, and FFFF the lowest 2 bytes. Other tags
references are located in the master channel list associated with a
brick, cal, and channel name).
The data tag is 2 bytes and comes before the data which is 2 bytes Time
16 Bytes (FFFC HiHi-2bytes FFFD HiLow-2bytes FFFE LowHi-2bytes FFFF
LowLow-2bytes) + DataTag1-2bytes Data1-2bytes + DataTag2-2bytes
Data2-2bytes +... until time starts over and then repeats until end
ofpacket.
The 2-bytes of data is a 16 bit integer +32764, -32768. Actually it is a
14 bit integer shifted to the left to make it a 16 bit integer. All cals
are done on the resulting 16 bit integer.

Sorry if this is a bit confusing. I will have a better more descriptive
write up out shortly. I will get it to you when I finish it (later this
week). If you have any questions please give me a holler.

Regards,

Matthew Powell
Naval Surface Warfare Center Carderock
Code 5500 Seakeeping
*************************************************/




class xcraftpacket{
public:
	xcraftpacket();//default constructor
	void swap2b(char*);//swaps bytes in 2-byte array
	void swap24b(char*);//swaps bytes in each 2-byte pair of 4-byte array
	bool readpacket(ifstream&,unsigned int);//reads a single packet and returns true if !eof()
	bool readtagchannel();//reads tagchannel.csv for tag number, channel name lookup table
	void printoutput();//prints ascii output data

	//binary array pointers
	//1 ascii character = 1 byte
	char* packetsize;
	char* flags;
	char* destip;
	char* srcip;
	char* port;
	char* fmtid;
	char* nodeid;
	char* paircount;
	char* packetcount;
	char* datatag;
	char* data;


	//stores tag number, channel name lookup table
	map<unsigned short, string> tagchannel;
	//stores tag number, scale factor lookup table
	map<unsigned short, float> tagscalefactor;


	//data storage map
	//stores sample according to data tag
	//1 datatag: many data samples
	map<unsigned short,vector<short> > datamatrix;
	map<unsigned short,vector<short> >::iterator dmi;
};



xcraftpacket::xcraftpacket(){
	packetsize = new char[2];
	flags = new char[2];
	destip = new char[4];
	srcip = new char[4];
	port = new char[2];
	fmtid = new char[2];
	nodeid = new char[2];
	paircount = new char[2];
	packetcount = new char[4];
	datatag = new char[2];
	data = new char[2];
}//end xcraftpacket



bool xcraftpacket::readtagchannel(){
	ifstream tc;
	tc.open("tagchannel.csv", ios::in);
	if(!tc.good()) return false;
	string tagstring;
	unsigned short tag = 0;
	string channelname;
	string scalefactorstring;
	float scalefactor = 0;
	while(!tc.eof()){
		tagstring.erase();
		channelname.erase();
		scalefactorstring.erase();
		getline(tc,tagstring,'\t');
		tag = atoi(tagstring.data());
		getline(tc,channelname,'\t');
		getline(tc,scalefactorstring,'\n');
		if(!tc.eof())
			tagchannel[tag] = channelname;
		scalefactor = atof(scalefactorstring.data());
		if(!tc.eof())
			tagscalefactor[tag] = scalefactor;
	}//end while
	tc.close();

	return true;
}//end readtagchannel



void xcraftpacket::swap2b(char* bb){
	char c = bb[0];
	bb[0] = bb[1];
	bb[1] = c;
}//end swap2b



void xcraftpacket::swap24b(char* bbbb){
	char c = bbbb[0];
	bbbb[0] = bbbb[1];
	bbbb[1] = c;
	c = bbbb[2];
	bbbb[2] = bbbb[3];
	bbbb[3] = c;
}//end swap2b



bool xcraftpacket::readpacket(ifstream& infile, unsigned int datafile){

	if(!infile.good())return false;//return false if the input file is not readable
	if(infile.eof())return false;//return false if end of input file

	//read packet header information
	infile.read(packetsize, 2);
	swap2b(packetsize);
	//if packetsize is not in proper range, stop
	while((*(unsigned short*)packetsize > 1500 || *(unsigned short*)packetsize < 700) && !infile.eof()){
		cerr << "error: datafile: " << datafile << " packetsize: " << *(unsigned short*)packetsize << " position: " << infile.tellg() << endl;
		return false;
	}//end while
	infile.read(flags, 2);
	swap2b(flags);
	infile.read(destip, 4);
	swap24b(destip);
	infile.read(srcip, 4);
	swap24b(srcip);
	infile.read(port, 2);
	swap2b(port);
	infile.read(fmtid, 2);
	swap2b(fmtid);
	infile.read(nodeid, 2);
	swap2b(nodeid);
	infile.read(paircount, 2);
	swap2b(paircount);
	infile.read(packetcount, 4);
	swap24b(packetcount);


	//read all pairs of datatag, data in the packet
	for(int d=0; d < *(unsigned short*)paircount && !infile.eof(); d++){
		//read a single datatag, data pair
		infile.read(datatag, 2);
		swap2b(datatag);
		infile.read(data, 2);
		swap2b(data);

		//store the datatag, data pair into the data storage map
		//do not concatenate time channels
		if(datafile == 1 || (*(unsigned short*)datatag != 0xfffc && *(unsigned short*)datatag != 0xfffd && *(unsigned short*)datatag != 0xfffe && *(unsigned short*)datatag != 0xffff))
			datamatrix[*(unsigned short*)datatag].insert(datamatrix[*(unsigned short*)datatag].end(),*(short*)data);
	}//end for

	if(infile.eof())return false;
	return true;
}//end readpacket



void xcraftpacket::printoutput(){
	//status output and error checking
	cout << "datamatrixsize: " << datamatrix.size() << endl;
	unsigned short tag = 0;
	unsigned int samples = 0;
	unsigned int maxsamples = 0;
	unsigned int minsamples = 0;
	cout << "tag samples: ";
	for(dmi=datamatrix.begin();dmi != datamatrix.end(); dmi++){
		tag = dmi->first;
		samples = dmi->second.size();
		cout << samples << " ";

		if(minsamples == 0){
			maxsamples = samples;
			minsamples = samples;
		}//end if
		if(samples > maxsamples) maxsamples = samples;
		else if(samples < minsamples) minsamples = samples;
	}//end for
	cout << endl;
	if(maxsamples != minsamples){
		cerr << "error: sample count mismatch: max: " << maxsamples << " min: " << minsamples << " delta: " << maxsamples-minsamples << endl;
	}//end if


	//output
	for(dmi=datamatrix.begin();dmi != datamatrix.end(); dmi++){
		tag = dmi->first;
		cout << "tag" << tag << "\t";
	}//end for
	cout << endl;
	string channelname;
	for(dmi=datamatrix.begin();dmi != datamatrix.end(); dmi++){
		tag = dmi->first;
		channelname.erase();
		channelname = tagchannel[tag];
		if(channelname.size() == 0)
			cout << "undefinedtag" << tag << "\t";
		else
			cout << channelname << "\t";
	}//end for
	cout << endl;
	float scalefactor = 1;
	float datasample = 0;
	for(unsigned int sample=0; sample < minsamples; sample++){
		for(dmi=datamatrix.begin();dmi != datamatrix.end(); dmi++){
			tag = dmi->first;
			scalefactor = tagscalefactor[tag];
			if(scalefactor == 0)scalefactor = 1;
			datasample = (dmi->second)[sample]*scalefactor;
			cout << datasample << "\t";
		}//end for
		cout << endl;
	}//end for
}//end printoutput





void printhelp(unsigned int argc, char* argv[]){
	//display help
	bool needhelp = false;
	if(argc <= 1)
		needhelp = true;
	else if(argc <= 1 || string(argv[1]) == "-help" || string(argv[1]) == "--help" || string(argv[1]) == "-h")
		needhelp = true;
	if(needhelp){
		cout << "\nGregory Alan Hildstrom\n";
		cout << "Hildstrom Engineering\n";
		cout << "http://www.hildstrom.com\n\n";
		cout << "This program treats every file in the\n";
		cout << "argument list as a separate set of channels for the\n";
		cout << "same data run and builds a single multiple-channel\n";
		cout << "ASCII tab-delimited data file. This utility\n";
		cout << "writes to the standard output, which can be\n";
		cout << "redirected to a file if desired.\n";
		cout << "This utility needs tagchannel.csv, which\n";
		cout << "is a 3-column tab-delimited ascii\n";
		cout << "text file that is used to look up\n";
		cout << "channel names and scale factors from data tag numbers.\n";
		cout << "tagchannel.csv must be in the same folder as\n";
		cout << "the exe file and have one tag<tab>channelname<tab>scalefactor\n";
		cout << "entry per line.\n\n";

		cout << "Usage:\n";
		cout << "xcdata                           Help\n";
		cout << "xcdata -h                        Help\n";
		cout << "xcdata -help                     Help\n";
		cout << "xcdata --help                    Help\n";
		cout << "xcdata iflist          Joins data files\n";
		cout << "xcdata *.dat           Example Joins all .dat data files\n";
		cout << "iflist like *.dat                     Input file list can use wildcards\n";
		cout << "iflist like *Sensor_1_* *Sensor_2_*   Input file list can use wildcards\n";

		cout << "\n";

	}//end if
}//end printhelp








int main(unsigned int argc, char* argv[]){
	cout << "XCraft Data Utility.\n";
	cout << "version: 2005-12-01-15:31\n";
	cout << "-----------------------------------\n";
	cout << "There were " << argc << " arguments.\n";

	printhelp(argc,argv);

	xcraftpacket xcp;
	if(!xcp.readtagchannel()){
		cerr << "error: problem with tagchannel.csv\n";
		return 1;
	}//end if


	cout << "number of data files: " << argc-1 << endl;
	cout << "reading files: ";
	ifstream infile;
	for(unsigned int datafile=1; datafile < argc; datafile++){
		cout << argv[datafile] << " ";

		//open next data file
		infile.clear();
		infile.open(argv[datafile],ios::in | ios::binary);

		//read data until file is finished
		while(xcp.readpacket(infile,datafile));

		//close data file
		infile.close();

		cout << "done ";
	}//end for
	cout << endl;


	xcp.printoutput();

	return 0;
}//end main

